
# Edit Record
# Copyright 2008 by Alexander V. Christensen

"""
Edit the currently selected row using a form-like interface.
"""

#    This file is part of GanttPV.
#
#    GanttPV is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    GanttPV is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with GanttPV; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# 081009 - first version
# 081126 - Brian - set minimum column width for multi-line; enable tabbing
# 090120 - Brian - show only matching columns in two-column reports
# 090120 - Alex - added logic for Report Names in report 1 (the main report)

def hint(s):
    try:
        Data.Hint("%s: %s" % (scriptname, s))
    except AttributeError:
        self.SetStatusText(s)

class EditRecordDialog(wx.Dialog):
    def __init__(self, parent, reportid, rowid):
        wx.Dialog.__init__(self, parent, -1, 'Edit Record', style = wx.DEFAULT_FRAME_STYLE)
        self.reportid = reportid
        self.rowid = rowid
        self.clist = Data.GetColumnList(reportid)

        border_sizer = wx.BoxSizer(wx.HORIZONTAL)
        scroller = wx.ScrolledWindow(self, style = wx.SUNKEN_BORDER|wx.TAB_TRAVERSAL)
        sizer = wx.BoxSizer(wx.VERTICAL)
        mini_border_sizer = wx.BoxSizer(wx.HORIZONTAL)
        grid_sizer = wx.FlexGridSizer(0, 2, hgap=15, vgap=10)

        row_table = Data.Database['ReportRow'][rowid].get('TableName')
        report = Data.Database['Report'][reportid]
        report_type_id = report.get('ReportTypeID')
        report_type = Data.Database['ReportType'][report_type_id]
        if row_table == report_type.get('TableA'):
            col_select = 'A'
        elif row_table == report_type.get('TableB'):
            col_select = 'B'
        else:
            col_select = '-'

        self.ctrls = []
        self.colids = []
        for colid in self.clist:
            rc = Data.ReportColumn[colid]
            ctid = rc.get('ColumnTypeID')
            ct = Data.ColumnType[ctid]
            if ct.get('T') not in (col_select, 'X'):
                continue
            
            label = wx.StaticText(scroller, -1, Data.GetColumnHeader(colid, -1))
            grid_sizer.Add(label, 0) # , wx.ALIGN_CENTER_VERTICAL)

            editable = ct.get('Edit') and not rc.get('Periods')
            value = Data.GetCellValue(rowid, colid, -1)

            if editable:
                width_in_grid = (rc.get('Width') or ct.get('Width') or 40)
                width = width_in_grid * 1.2
                height = 25
                style = 0

                if ct.get('DataType') == 't' and width_in_grid > 150:
                    width += 20
                    height += 40
                    style = wx.TE_MULTILINE

                ctrl = wx.TextCtrl(scroller, -1, size=(width, height), style=style)
                ctrl.SetValue(str(value))
            else:
                ctrl = wx.StaticText(scroller, -1)
                ctrl.SetLabel(str(value))

            self.ctrls.append(ctrl)
            self.colids.append(colid)
            grid_sizer.Add(ctrl, 0, wx.ALIGN_LEFT)

        mini_border_sizer.Add(grid_sizer, 1, wx.ALL, 10)
        scroller.SetSizer(mini_border_sizer)
        scroller.Fit()
        if scroller.GetSize()[1] > 400: scroller.SetSize((-1, 400))
        scroller.SetScrollRate(20, 20)
        sizer.Add(scroller, 1, wx.GROW)
        sizer.Add((10, 10))

        buttons = wx.BoxSizer(wx.HORIZONTAL)
        cancelbutton = wx.Button(self, wx.ID_CANCEL, "Cancel", size=(70, 20))
        buttons.Add(cancelbutton, 0, wx.ALIGN_CENTER_VERTICAL)
        buttons.Add((10, 10))
        okbutton = wx.Button(self, wx.ID_OK, "OK", size=(70, 20))
        okbutton.SetDefault()
        buttons.Add(okbutton, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.Add(buttons, 0, wx.ALIGN_RIGHT)

        border_sizer.Add(sizer, 1, wx.ALL | wx.ALIGN_CENTRE | wx.GROW, 10)
        self.SetSizer(border_sizer)
        self.Fit()
        self.CentreOnScreen()

        wx.EVT_BUTTON(self, wx.ID_OK, self.OnOK)

    def OnOK(self, event):
        for colid, ctrl in zip(self.colids, self.ctrls):
            try:
                value = ctrl.GetValue()
                Data.SetCellValue(self.rowid, colid, -1, value)
            except AttributeError:
                pass
        Data.SetUndo("Edit Record")
        event.Skip()

def Do():
    if self.ReportID == 1:
        rowid = self.Report.rows[self.Report.currentItem or 0]
        
        rr = Data.ReportRow[rowid]
        table = rr['TableName']
        if table == 'Report':  # if report, just edit the report name
            id = rr['TableID']
            dlg = wx.TextEntryDialog(None, 'New report name', 'Edit Report Name', '')
            oldvalue = Data.Report[id].get('Name', "")
            dlg.SetValue(oldvalue)
            if dlg.ShowModal() == wx.ID_OK:
                change = { 'Table': table, 'ID': id, 'Name': dlg.GetValue() }
                Data.Update(change)
                Data.SetUndo('Edit Record')
            return
    else:
        if not self.Report.GetNumberRows():
            hint("No records to edit.")
            return
    
        sel = self.Report.GetSelectedRows()
        if sel:
            row = min(sel)
        else:
            row = self.Report.GetGridCursorRow()
        rowid = self.Report.table.rows[row]

    dlg = EditRecordDialog(None, self.ReportID, rowid)
    dlg.ShowModal()

Do()
